﻿using System;
using System.Text;
using System.Collections.Generic;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using PcapAnalyzer;

namespace PcapAnalyzerTest
{
    [TestClass]
    public class PasswordsModuleTest
    {
        [TestMethod]
        public void FtpPasswordParser_ParseFtpPassword_ParseSuccess()
        {
            // Arrange.
            var ftpParsrer = new PcapAnalyzer.FtpPasswordParser();
            var session = new PcapAnalyzer.TcpSession();
            session.SourceIp = "1.1.1.1";
            session.DestinationIp = "2.2.2.2";
            session.Data = Encoding.UTF8.GetBytes(
@"220 Chris Sanders FTP Server
USER csanders
331 Password required for csanders.
PASS echo
230 User csanders logged in.");

            // Act.
            PcapAnalyzer.NetworkPassword passsword = (ftpParsrer.Parse(session) as PcapAnalyzer.NetworkPassword);

            // Assert.
            Assert.AreEqual("csanders", passsword.Username);
            Assert.AreEqual("echo", passsword.Password);
        }

        [TestMethod]
        public void HttpBasicPasswordParser_ParseFtpPassword_ParseSuccess()
        {
            // Arrange.
            var parsrer = new PcapAnalyzer.HttpBasicPasswordParser();
            var packet = new PcapAnalyzer.TcpPacket();
            packet.SourceIp = "1.1.1.1";
            packet.DestinationIp = "2.2.2.2";
            packet.Data = Encoding.UTF8.GetBytes(
@"GET /password-ok.php HTTP/1.1
Host: browserspy.dk
Connection: keep-alive
Cache-Control: max-age=0
Authorization: Basic dGVzdDpmYWlsMw==
Accept: text/html,application/xhtml+xml");

            // Act.
            PcapAnalyzer.NetworkPassword password = (parsrer.Parse(packet) as PcapAnalyzer.NetworkPassword);

            // Assert.
            Assert.AreEqual("test", password.Username);
            Assert.AreEqual("fail3", password.Password);
        }

        [TestMethod]
        public void TelnetPasswordParser_ParseTelnetCharModePassword_ParseSuccess()
        {
            // Arrange
            var telnetParser = new PcapAnalyzer.TelnetPasswordParser();
            var session = new PcapAnalyzer.TcpSession();

            var expected = new PcapAnalyzer.NetworkPassword()
            {
                Username = "us",
                Password = "Secret123",
                Destination = "2.2.2.2",
                Source = "1.1.1.1",
                Protocol = "Telnet"
            };

            // Mock a session where the user is "us" and the password is "Secret123".
            session.Data = Encoding.ASCII.GetBytes("login:uuss" + Environment.NewLine + "Password:Secret123");
            session.Packets.Add(mockPacket("2.2.2.2", "1.1.1.1", 21, 5472, "login:"));
            session.Packets.Add(mockPacket("1.1.1.1", "2.2.2.2", 5472, 21, "u"));
            session.Packets.Add(mockPacket("2.2.2.2", "1.1.1.1", 21, 5472, "u"));
            session.Packets.Add(mockPacket("1.1.1.1", "2.2.2.2", 5472, 21, "s"));
            session.Packets.Add(mockPacket("2.2.2.2", "1.1.1.1", 21, 5472, "s"));
            session.Packets.Add(mockPacket("2.2.2.2", "1.1.1.1", 21, 5472, "Password:"));
            session.Packets.Add(mockPacket("1.1.1.1", "2.2.2.2", 5472, 21, "Secret123"));
            session.Packets.Add(mockPacket("2.2.2.2", "1.1.1.1", 21, 5472, "some dummy data"));

            // Act.
            PcapAnalyzer.NetworkPassword password = (telnetParser.Parse(session) as PcapAnalyzer.NetworkPassword);

            // Assert.
            Assert.AreEqual(expected, password);

        }

        [TestMethod]
        public void NtlmPasswordParser_ParseSmbNTLMv2Session_ParseSuccess()
        {
            // Arrange
            var ntlmParser = new PcapAnalyzer.NtlmsspHashParser();
            var session = new PcapAnalyzer.TcpSession();

            var serverPacket = new PcapAnalyzer.TcpPacket()
            {
                SourceIp = "2.2.2.2",
                DestinationIp = "1.1.1.1",
                Data = new byte[]
                {
                    0x00, 0x00, 0x00, 0xf8, 0xfe, 0x53, 0x4d, 0x42, 0x40, 0x00, 0x00, 0x00, 0x16, 0x00, 0x00, 0xc0,
                    0x01, 0x00, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x94,
                    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x48, 0x00, 0xb0, 0x00, 0x4e, 0x54, 0x4c, 0x4d,
                    0x53, 0x53, 0x50, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00, 0x38, 0x00, 0x00, 0x00,
                    0x35, 0x02, 0x89, 0xe2, 0x01, 0x15, 0x18, 0x13, 0xd2, 0x89, 0x8c, 0xcd, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 0x70, 0x00, 0x40, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x39, 0x38,
                    0x00, 0x00, 0x00, 0x0f, 0x53, 0x00, 0x55, 0x00, 0x53, 0x00, 0x45, 0x00, 0x02, 0x00, 0x08, 0x00,
                    0x53, 0x00, 0x55, 0x00, 0x53, 0x00, 0x45, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x57, 0x00, 0x53, 0x00,
                    0x32, 0x00, 0x30, 0x00, 0x31, 0x00, 0x36, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x73, 0x00, 0x75, 0x00,
                    0x73, 0x00, 0x65, 0x00, 0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x03, 0x00, 0x1c, 0x00, 0x57, 0x00,
                    0x53, 0x00, 0x32, 0x00, 0x30, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x73, 0x00, 0x75, 0x00,
                    0x73, 0x00, 0x65, 0x00, 0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x05, 0x00, 0x0e, 0x00, 0x73, 0x00,
                    0x75, 0x00, 0x73, 0x00, 0x65, 0x00, 0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x07, 0x00, 0x08, 0x00,
                    0x8a, 0x8c, 0xe7, 0xa9, 0xf4, 0xce, 0xd2, 0x01, 0x00, 0x00, 0x00, 0x00
                }
            };

            var clientPacket = new PcapAnalyzer.TcpPacket()
            {
                SourceIp = "1.1.1.1",
                DestinationIp = "2.2.2.2",
                Data = new byte[]
                {
                    0x00, 0x00, 0x01, 0x68, 0xfe, 0x53, 0x4d, 0x42, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x01, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0xd5, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3d, 0x00, 0x00, 0x94,
                    0x00, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x19, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
                    0x58, 0x00, 0x10, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4e, 0x54, 0x4c, 0x4d,
                    0x53, 0x53, 0x50, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00,
                    0x9c, 0x00, 0x9c, 0x00, 0x40, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00, 0xdc, 0x00, 0x00, 0x00,
                    0x1a, 0x00, 0x1a, 0x00, 0xe4, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00,
                    0x10, 0x00, 0x10, 0x00, 0x00, 0x01, 0x00, 0x00, 0x35, 0x02, 0x88, 0xe0, 0x39, 0xdb, 0xdb, 0xeb,
                    0x1b, 0xdd, 0x29, 0xb0, 0x7a, 0x5d, 0x20, 0xc8, 0xf8, 0x2f, 0x2c, 0xb7, 0x01, 0x01, 0x00, 0x00,
                    0x00, 0x00, 0x00, 0x00, 0x8a, 0x8c, 0xe7, 0xa9, 0xf4, 0xce, 0xd2, 0x01, 0xe7, 0x96, 0x9a, 0x04,
                    0x87, 0x2c, 0x16, 0x89, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x08, 0x00, 0x53, 0x00, 0x55, 0x00,
                    0x53, 0x00, 0x45, 0x00, 0x01, 0x00, 0x0c, 0x00, 0x57, 0x00, 0x53, 0x00, 0x32, 0x00, 0x30, 0x00,
                    0x31, 0x00, 0x36, 0x00, 0x04, 0x00, 0x0e, 0x00, 0x73, 0x00, 0x75, 0x00, 0x73, 0x00, 0x65, 0x00,
                    0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x03, 0x00, 0x1c, 0x00, 0x57, 0x00, 0x53, 0x00, 0x32, 0x00,
                    0x30, 0x00, 0x31, 0x00, 0x36, 0x00, 0x2e, 0x00, 0x73, 0x00, 0x75, 0x00, 0x73, 0x00, 0x65, 0x00,
                    0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x05, 0x00, 0x0e, 0x00, 0x73, 0x00, 0x75, 0x00, 0x73, 0x00,
                    0x65, 0x00, 0x2e, 0x00, 0x64, 0x00, 0x65, 0x00, 0x07, 0x00, 0x08, 0x00, 0x8a, 0x8c, 0xe7, 0xa9,
                    0xf4, 0xce, 0xd2, 0x01, 0x00, 0x00, 0x00, 0x00, 0x53, 0x00, 0x55, 0x00, 0x53, 0x00, 0x45, 0x00,
                    0x61, 0x00, 0x64, 0x00, 0x6d, 0x00, 0x69, 0x00, 0x6e, 0x00, 0x69, 0x00, 0x73, 0x00, 0x74, 0x00,
                    0x72, 0x00, 0x61, 0x00, 0x74, 0x00, 0x6f, 0x00, 0x72, 0x00, 0x00, 0x00, 0xb2, 0xe8, 0x76, 0x55,
                    0x9c, 0x9c, 0x58, 0xb0, 0x34, 0x4b, 0xd5, 0xa9, 0x9f, 0x8e, 0x98, 0x55
                }
            };

            session.Packets.Add(serverPacket);
            session.Packets.Add(clientPacket);

            // Act.
            var hash = ntlmParser.Parse(session) as PcapAnalyzer.NtlmHash;

            // Assert.
            Assert.AreEqual("NTLMSSP", hash.Protocol);
            Assert.AreEqual("administrator", hash.User);
            Assert.AreEqual("SUSE", hash.Domain);
            Assert.AreEqual(hash.NtHash.Length, 312);
        }

        [TestMethod]
        public void KerberosHashParser_ParseUdpSession_ParseSuccess()
        {
            // Arrange
            var kerberosParser = new PcapAnalyzer.KerberosAsReqHashParser();

            var kerberosAsRequestPacket = new PcapAnalyzer.UdpPacket
            {
                SourceIp = "2.2.2.2",
                DestinationIp = "1.1.1.1",
                Data = new byte[]
                {
                    0x6a, 0x82, 0x01, 0x1f, 0x30, 0x82, 0x01, 0x1b, 0xa1, 0x03, 0x02, 0x01, 0x05, 0xa2, 0x03, 0x02,
                    0x01, 0x0a, 0xa3, 0x5f, 0x30, 0x5d, 0x30, 0x48, 0xa1, 0x03, 0x02, 0x01, 0x02, 0xa2, 0x41, 0x04,
                    0x3f, 0x30, 0x3d, 0xa0, 0x03, 0x02, 0x01, 0x17, 0xa2, 0x36, 0x04, 0x34, 0x09, 0xa2, 0x24, 0x48,
                    0x93, 0xaf, 0xf5, 0xf3, 0x84, 0xf7, 0x9c, 0x37, 0x88, 0x3f, 0x15, 0x4a, 0x32, 0xd3, 0x96, 0xa9,
                    0x14, 0xa4, 0xd0, 0xa7, 0x8e, 0x97, 0x9b, 0xa7, 0x5d, 0x4f, 0xf5, 0x3c, 0x1d, 0xb7, 0x29, 0x41,
                    0x41, 0x76, 0x0f, 0xee, 0x05, 0xe4, 0x34, 0xc1, 0x2e, 0xcf, 0x8d, 0x5b, 0x9a, 0xa5, 0x83, 0x9e,
                    0x30, 0x11, 0xa1, 0x04, 0x02, 0x02, 0x00, 0x80, 0xa2, 0x09, 0x04, 0x07, 0x30, 0x05, 0xa0, 0x03,
                    0x01, 0x01, 0xff, 0xa4, 0x81, 0xad, 0x30, 0x81, 0xaa, 0xa0, 0x07, 0x03, 0x05, 0x00, 0x40, 0x81,
                    0x00, 0x10, 0xa1, 0x10, 0x30, 0x0e, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x07, 0x30, 0x05, 0x1b,
                    0x03, 0x64, 0x65, 0x73, 0xa2, 0x08, 0x1b, 0x06, 0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0xa3, 0x1b,
                    0x30, 0x19, 0xa0, 0x03, 0x02, 0x01, 0x02, 0xa1, 0x12, 0x30, 0x10, 0x1b, 0x06, 0x6b, 0x72, 0x62,
                    0x74, 0x67, 0x74, 0x1b, 0x06, 0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0xa5, 0x11, 0x18, 0x0f, 0x32,
                    0x30, 0x33, 0x37, 0x30, 0x39, 0x31, 0x33, 0x30, 0x32, 0x34, 0x38, 0x30, 0x35, 0x5a, 0xa6, 0x11,
                    0x18, 0x0f, 0x32, 0x30, 0x33, 0x37, 0x30, 0x39, 0x31, 0x33, 0x30, 0x32, 0x34, 0x38, 0x30, 0x35,
                    0x5a, 0xa7, 0x06, 0x02, 0x04, 0x0b, 0xc4, 0xdd, 0x7e, 0xa8, 0x19, 0x30, 0x17, 0x02, 0x01, 0x17,
                    0x02, 0x02, 0xff, 0x7b, 0x02, 0x01, 0x80, 0x02, 0x01, 0x03, 0x02, 0x01, 0x01, 0x02, 0x01, 0x18,
                    0x02, 0x02, 0xff, 0x79, 0xa9, 0x1d, 0x30, 0x1b, 0x30, 0x19, 0xa0, 0x03, 0x02, 0x01, 0x14, 0xa1,
                    0x12, 0x04, 0x10, 0x58, 0x50, 0x31, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
                    0x20, 0x20, 0x20
                }
            };

            // Act.
            var hash = kerberosParser.Parse(kerberosAsRequestPacket) as PcapAnalyzer.KerberosHash;

            // Assert.
            Assert.AreEqual("Kerberos V5 AS-REQ Pre-Auth etype 23", hash.HashType);
            Assert.AreEqual("des", hash.User);
            Assert.AreEqual("DENYDC", hash.Domain);
            Assert.AreEqual(hash.Hash, "32d396a914a4d0a78e979ba75d4ff53c1db7294141760fee05e434c12ecf8d5b9aa5839e09a2244893aff5f384f79c37883f154a");
        }

        [TestMethod]
        public void KerberosTgsRepHashOverUdp_Parse_ParseSuccess()
        {
            // Arrange.
            var kerberosTgsRepBuffer = new byte[]
            {
                0x6d, 0x82, 0x05, 0x08, 0x30, 0x82,
                0x05, 0x04, 0xa0, 0x03, 0x02, 0x01, 0x05, 0xa1, 0x03, 0x02, 0x01, 0x0d, 0xa3, 0x0c, 0x1b, 0x0a,
                0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0x2e, 0x43, 0x4f, 0x4d, 0xa4, 0x1a, 0x30, 0x18, 0xa0, 0x03,
                0x02, 0x01, 0x01, 0xa1, 0x11, 0x30, 0x0f, 0x1b, 0x0d, 0x41, 0x64, 0x6d, 0x69, 0x6e, 0x69, 0x73,
                0x74, 0x72, 0x61, 0x74, 0x6f, 0x72, 0xa5, 0x82, 0x03, 0xce, 0x61, 0x82, 0x03, 0xca, 0x30, 0x82,
                0x03, 0xc6, 0xa0, 0x03, 0x02, 0x01, 0x05, 0xa1, 0x0c, 0x1b, 0x0a, 0x44, 0x45, 0x4e, 0x59, 0x44,
                0x43, 0x2e, 0x43, 0x4f, 0x4d, 0xa2, 0x27, 0x30, 0x25, 0xa0, 0x03, 0x02, 0x01, 0x02, 0xa1, 0x1e,
                0x30, 0x1c, 0x1b, 0x04, 0x48, 0x54, 0x54, 0x50, 0x1b, 0x14, 0x75, 0x70, 0x77, 0x69, 0x6e, 0x32,
                0x30, 0x30, 0x33, 0x2e, 0x64, 0x65, 0x6e, 0x79, 0x44, 0x43, 0x2e, 0x63, 0x6f, 0x6d, 0xa3, 0x82,
                0x03, 0x86, 0x30, 0x82, 0x03, 0x82, 0xa0, 0x03, 0x02, 0x01, 0x17, 0xa1, 0x03, 0x02, 0x01, 0x02,
                0xa2, 0x82, 0x03, 0x74, 0x04, 0x82, 0x03, 0x70, 0xb5, 0x58, 0x84, 0x7e, 0x2a, 0xa7, 0xf0, 0xd2,
                0x7c, 0xd9, 0x46, 0x52, 0xc7, 0x00, 0xfe, 0xb4, 0xad, 0x49, 0x2a, 0xb8, 0x8e, 0x5f, 0xec, 0x0f,
                0xea, 0x18, 0x2d, 0xd7, 0x55, 0xa2, 0xdb, 0x8b, 0x20, 0xd5, 0xdc, 0x5a, 0x69, 0x35, 0xcb, 0x4b,
                0x18, 0xf2, 0x05, 0xaf, 0xf7, 0xd5, 0x6f, 0x2b, 0x2f, 0x63, 0x2d, 0xcc, 0x59, 0x3e, 0x64, 0xc3,
                0x0f, 0x1d, 0xce, 0xb8, 0x06, 0x06, 0xb5, 0xaf, 0x6d, 0xf2, 0x55, 0xe1, 0x5b, 0x7a, 0x03, 0xba,
                0x35, 0x1f, 0x08, 0x3c, 0xf3, 0xf3, 0x70, 0xdc, 0x07, 0xa3, 0xf2, 0x2f, 0x6c, 0x05, 0xa2, 0xa6,
                0xa7, 0xd5, 0x06, 0x3f, 0x2e, 0xfd, 0x7c, 0x39, 0x34, 0x89, 0xe3, 0x1a, 0xd1, 0x57, 0xd6, 0xfb,
                0xd2, 0xd5, 0xa6, 0x36, 0x3a, 0x30, 0xf1, 0x6e, 0x71, 0xb2, 0x1b, 0x34, 0x10, 0x70, 0x2f, 0x92,
                0x42, 0x48, 0x2e, 0x3d, 0x8d, 0xb8, 0xdd, 0x5f, 0xe4, 0x9c, 0x20, 0x6a, 0x23, 0x15, 0xfa, 0x5c,
                0x7b, 0xbe, 0xf2, 0xcc, 0x9d, 0x1d, 0x31, 0xe1, 0x35, 0x22, 0x97, 0xf1, 0x90, 0xb3, 0x66, 0x8f,
                0x4a, 0x62, 0xb2, 0x7c, 0x04, 0xf8, 0x5c, 0x77, 0x29, 0x5a, 0x1f, 0x66, 0xdb, 0xd7, 0x1c, 0x83,
                0x9a, 0xf4, 0x62, 0x54, 0xcc, 0xbb, 0x64, 0x51, 0x2d, 0xdc, 0xa4, 0xf9, 0x72, 0xb3, 0x7c, 0xd4,
                0xc3, 0x9f, 0xf3, 0xc9, 0x2b, 0x1f, 0xe2, 0x84, 0x19, 0xc5, 0x31, 0x6d, 0x79, 0x7a, 0x00, 0x11,
                0x31, 0x74, 0xd8, 0x00, 0xf3, 0x1e, 0x0f, 0xb9, 0xf5, 0x77, 0xc6, 0xba, 0x1f, 0x31, 0xe7, 0x32,
                0xff, 0x94, 0xa0, 0x7e, 0x8b, 0x15, 0xaa, 0xfb, 0x90, 0x29, 0x40, 0x03, 0x2e, 0x96, 0x90, 0x2b,
                0xf3, 0x71, 0x11, 0xa4, 0x9f, 0x18, 0xbc, 0x6c, 0x47, 0xf6, 0x34, 0x23, 0x4a, 0x48, 0xa4, 0x86,
                0xea, 0x9e, 0xcd, 0xbf, 0xa0, 0x99, 0x9f, 0xe9, 0x92, 0xe5, 0xca, 0x9d, 0xab, 0x49, 0xfc, 0x8a,
                0xac, 0x19, 0x23, 0x44, 0x02, 0x6e, 0x54, 0x1f, 0xf1, 0x80, 0x50, 0xc0, 0x40, 0x6d, 0x16, 0x57,
                0x40, 0xce, 0xf2, 0x85, 0x06, 0x6e, 0x9d, 0xcb, 0x1b, 0x5d, 0x99, 0xca, 0x22, 0x6c, 0x0b, 0x54,
                0x70, 0x29, 0xfb, 0x7a, 0x51, 0xaf, 0xff, 0x8b, 0x65, 0x70, 0x9d, 0x44, 0xf7, 0x5d, 0x76, 0x7c,
                0x69, 0x9f, 0x31, 0xa2, 0x48, 0x59, 0xe5, 0x2c, 0x45, 0x6f, 0x74, 0x4f, 0xbb, 0xed, 0xf8, 0xa5,
                0x74, 0x78, 0x72, 0x3d, 0x4c, 0x00, 0xf1, 0xa4, 0xf1, 0x33, 0x3d, 0x09, 0xd5, 0x91, 0x70, 0x37,
                0xcc, 0x28, 0x37, 0xe0, 0x92, 0x61, 0x02, 0x13, 0xd0, 0x6f, 0x66, 0xee, 0x32, 0x7d, 0x03, 0x34,
                0x9f, 0xff, 0xb0, 0x44, 0x56, 0xa0, 0xab, 0xf4, 0x9c, 0xfb, 0xa3, 0x40, 0x88, 0x21, 0xf0, 0x21,
                0x85, 0x85, 0x45, 0x5b, 0x3b, 0xf3, 0x23, 0xfa, 0xdb, 0x5f, 0x74, 0xf3, 0x05, 0xf1, 0xf0, 0x74,
                0xd3, 0x74, 0xfb, 0x8a, 0xea, 0xdc, 0x41, 0x24, 0x9a, 0x1d, 0xcc, 0x85, 0x15, 0xc5, 0x56, 0x57,
                0x34, 0x73, 0x53, 0x1d, 0x5b, 0x23, 0xb5, 0x58, 0xae, 0x3e, 0xcb, 0xee, 0x73, 0x2b, 0x86, 0xd1,
                0xec, 0xc9, 0x4b, 0x33, 0x6a, 0x7e, 0xd4, 0xe2, 0x52, 0x3e, 0xbc, 0x24, 0x94, 0x21, 0xfe, 0x41,
                0x25, 0x6f, 0x2e, 0x9d, 0x1a, 0x03, 0x62, 0xfa, 0x6f, 0x50, 0x8d, 0xb1, 0xd8, 0xe9, 0x85, 0x16,
                0x95, 0xa3, 0xe9, 0xa9, 0xf1, 0xc1, 0x6d, 0x14, 0xda, 0x9a, 0xce, 0x11, 0x89, 0xbb, 0x44, 0x89,
                0xf6, 0xf9, 0x71, 0x86, 0x47, 0x3e, 0x75, 0x02, 0xf3, 0x97, 0x22, 0x0d, 0x53, 0x79, 0x7f, 0x8d,
                0xd4, 0x88, 0x4b, 0x58, 0xf9, 0x4b, 0x86, 0xc4, 0x2b, 0xdd, 0xb6, 0xc9, 0x79, 0xa6, 0x5d, 0x4c,
                0x6e, 0x0e, 0x80, 0x1f, 0x29, 0x03, 0x6b, 0x80, 0x33, 0x24, 0x94, 0x7e, 0x9a, 0x60, 0xc5, 0x69,
                0xf1, 0x1d, 0x28, 0x7c, 0xb9, 0x9e, 0xab, 0xce, 0x84, 0x9e, 0x18, 0xd3, 0xe3, 0x6c, 0x1c, 0xbd,
                0xe4, 0xba, 0x66, 0xec, 0x14, 0x4d, 0x72, 0xef, 0xcc, 0x49, 0xa0, 0xa6, 0xbf, 0x51, 0xa3, 0x3c,
                0x20, 0x27, 0xc9, 0x49, 0xee, 0x6e, 0x70, 0xf7, 0xae, 0xf7, 0x87, 0x69, 0x43, 0x32, 0xf1, 0xd0,
                0x33, 0x08, 0x9f, 0xc1, 0x20, 0x3e, 0x6c, 0xad, 0xdc, 0x5b, 0x0e, 0xb7, 0x8e, 0xfa, 0xe6, 0xa8,
                0xd3, 0xef, 0x7c, 0x28, 0xf2, 0x4c, 0x36, 0x0c, 0x7b, 0x06, 0x64, 0x6c, 0x8e, 0x14, 0x49, 0xe6,
                0x76, 0xda, 0x56, 0x39, 0xc6, 0xb3, 0xf9, 0x43, 0xba, 0xcf, 0x3a, 0x55, 0x67, 0x7d, 0x45, 0xb6,
                0x01, 0x16, 0x8c, 0x00, 0xff, 0xf4, 0xce, 0x27, 0x87, 0x42, 0x12, 0x04, 0x05, 0x53, 0x65, 0x02,
                0xd1, 0x0d, 0x6c, 0xac, 0x75, 0x3d, 0x14, 0xbc, 0x6e, 0xa9, 0xcc, 0xdb, 0xb5, 0x4c, 0xf2, 0xc7,
                0x10, 0x6e, 0xd1, 0xa6, 0xf0, 0x5c, 0xa4, 0x95, 0x9e, 0x73, 0x9e, 0x34, 0x97, 0x79, 0x8e, 0x08,
                0xf1, 0x60, 0x73, 0x28, 0x78, 0xfc, 0xf1, 0x85, 0xd6, 0xdf, 0x3f, 0x1d, 0xfd, 0x98, 0xf1, 0x7a,
                0x1e, 0x0f, 0xc6, 0xe5, 0x87, 0x89, 0x94, 0xc1, 0xaa, 0x6a, 0xed, 0xe8, 0xe1, 0x68, 0xe0, 0x53,
                0x22, 0xa1, 0xf4, 0x22, 0x18, 0xc2, 0x0b, 0xcf, 0x57, 0xb2, 0xbf, 0x59, 0xbe, 0x1c, 0xab, 0x16,
                0xea, 0x9c, 0x36, 0x4d, 0x29, 0x48, 0x32, 0x40, 0xf7, 0x80, 0x67, 0x87, 0x43, 0x30, 0x2e, 0x67,
                0x51, 0xfa, 0xbb, 0x72, 0x1e, 0x6d, 0xdf, 0x3d, 0xcb, 0xc4, 0x12, 0x97, 0x53, 0x3e, 0xba, 0x56,
                0x76, 0x79, 0x54, 0x4a, 0x60, 0x02, 0x5e, 0x37, 0x6c, 0xfa, 0x5b, 0x6e, 0x7b, 0x83, 0x03, 0x0a,
                0x95, 0x2e, 0x44, 0x1b, 0x5a, 0x68, 0xe0, 0x31, 0x84, 0x33, 0x05, 0x13, 0x2a, 0x17, 0x69, 0x67,
                0xbe, 0x2b, 0x08, 0x50, 0xe0, 0xdc, 0x70, 0x40, 0x1a, 0x3b, 0xc3, 0x9c, 0xe1, 0x59, 0x9a, 0xe4,
                0x52, 0x9b, 0x25, 0x68, 0xab, 0xcd, 0xaa, 0x8f, 0x1d, 0xcd, 0x61, 0xbf, 0x44, 0xe9, 0x5b, 0x2a,
                0x34, 0xcf, 0x83, 0x9e, 0xc6, 0xc0, 0x92, 0x83, 0x51, 0x2f, 0x55, 0xb8, 0x24, 0x5c, 0xf3, 0x7f,
                0x0a, 0x13, 0x9b, 0xeb, 0x94, 0x84, 0x20, 0xd4, 0xc3, 0x53, 0x4e, 0x96, 0xc4, 0xbc, 0xb4, 0x02,
                0xf0, 0x86, 0x4d, 0x04, 0xee, 0x33, 0x82, 0x54, 0xd6, 0xd4, 0x1c, 0x89, 0x1d, 0xd5, 0xc5, 0x8a,
                0x59, 0x27, 0x62, 0x97, 0x56, 0x17, 0x64, 0x98, 0x76, 0x86, 0xde, 0xb4, 0xf3, 0x08, 0xdc, 0x44,
                0x96, 0x60, 0xc3, 0x2c, 0x00, 0xf7, 0x7d, 0x3e, 0xa6, 0x81, 0xfb, 0x30, 0x81, 0xf8, 0xa0, 0x03,
                0x02, 0x01, 0x17, 0xa2, 0x81, 0xf0, 0x04, 0x81, 0xed, 0xd3, 0x37, 0x5a, 0x42, 0xc2, 0xf0, 0xc5,
                0x77, 0x0f, 0xa7, 0xfd, 0xa7, 0x12, 0xae, 0x49, 0x5d, 0x42, 0xfd, 0xc7, 0x3f, 0x7f, 0xe5, 0x9a,
                0xd0, 0xd9, 0x42, 0xc6, 0x13, 0x32, 0xbc, 0x8f, 0x17, 0xb4, 0x13, 0x4a, 0xd9, 0x4d, 0x20, 0xa9,
                0xe2, 0xc0, 0x7c, 0xc2, 0x16, 0x40, 0x31, 0xa6, 0xca, 0x28, 0xf0, 0x45, 0xa0, 0xe5, 0x72, 0xd2,
                0xb5, 0x0c, 0xdc, 0x84, 0x64, 0xcf, 0x13, 0x57, 0xab, 0xce, 0xbb, 0x83, 0x81, 0x15, 0x75, 0x88,
                0x2b, 0x66, 0xb5, 0x1e, 0xe6, 0x9c, 0x00, 0x50, 0x52, 0x5f, 0x45, 0xf2, 0xd1, 0x5b, 0x3a, 0xcc,
                0x16, 0xee, 0x2b, 0xa4, 0xc5, 0x17, 0xfb, 0x24, 0x82, 0x7e, 0xc5, 0x37, 0x48, 0x5f, 0x5d, 0xed,
                0xb0, 0x0f, 0x39, 0x31, 0x59, 0xb8, 0x4b, 0xc0, 0xda, 0x41, 0xdf, 0x04, 0xb2, 0x65, 0x11, 0x83,
                0x9d, 0x77, 0xed, 0x08, 0xac, 0xc7, 0xe7, 0x6b, 0xaf, 0x67, 0xfb, 0xc3, 0xd9, 0x44, 0xe7, 0x6d,
                0x62, 0x7e, 0xfc, 0x86, 0x02, 0xc8, 0xc8, 0x9f, 0x4a, 0x42, 0x3b, 0x0e, 0x72, 0x7f, 0x70, 0x5c,
                0x6c, 0xd4, 0x76, 0x57, 0x69, 0x30, 0x45, 0x9d, 0xca, 0xc2, 0xaf, 0x50, 0x7b, 0x07, 0x22, 0xde,
                0x66, 0x98, 0xce, 0x26, 0xf2, 0x37, 0x6d, 0x91, 0x05, 0x86, 0x42, 0xf0, 0x73, 0x85, 0x55, 0xe3,
                0x10, 0x74, 0x56, 0x1f, 0x21, 0x44, 0x69, 0xc6, 0x19, 0x05, 0x33, 0x8b, 0x17, 0xd6, 0x40, 0x98,
                0x60, 0x06, 0x1c, 0x36, 0xcf, 0x95, 0x5d, 0xe5, 0x8e, 0x4e, 0xb4, 0x56, 0x61, 0x90, 0x1f, 0xf2,
                0xfc, 0xf8, 0x32, 0xda, 0x1f, 0x15, 0x2e, 0x56, 0x45, 0x61, 0x25, 0x77, 0x47, 0x49, 0x11, 0x6a,
                0xb6, 0x93, 0xd2, 0x1e, 0x4d, 0xa3
            };

            // Act.
            var kerberosTgsRepPacket = KerberosPacketParser.GetKerberosPacket(kerberosTgsRepBuffer, protocol:"UDP") as KerberosTgsRepPacket;

            // Assert.
            Assert.AreEqual(kerberosTgsRepPacket.Pvno, 5);
            Assert.AreEqual(kerberosTgsRepPacket.Crealm, "DENYDC.COM");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Realm, "DENYDC.COM");
            Assert.AreEqual(kerberosTgsRepPacket.Cname.Name, "Administrator");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Sname.Name, "HTTP/upwin2003.denyDC.com");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Etype, 23);
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Cipher.Length, 880);
        }

        [TestMethod]
        public void KerberosTgsRepHashOverTcp_Parse_ParseSuccess()
        {
            // Arrange.
            var kerberosTgsRepBuffer = new byte[]
            {
                0x00, 0x00, 0x05, 0xbe, 0x6d, 0x82, 0x05, 0xba, 0x30, 0x82, 0x05, 0xb6, 0xa0, 0x03, 0x02, 0x01,
                0x05, 0xa1, 0x03, 0x02, 0x01, 0x0d, 0xa3, 0x0f, 0x1b, 0x0d, 0x43, 0x4f, 0x4e, 0x43, 0x41, 0x56,
                0x45, 0x2e, 0x4c, 0x4f, 0x43, 0x41, 0x4c, 0xa4, 0x10, 0x30, 0x0e, 0xa0, 0x03, 0x02, 0x01, 0x01,
                0xa1, 0x07, 0x30, 0x05, 0x1b, 0x03, 0x6a, 0x69, 0x6d, 0xa5, 0x82, 0x04, 0x4e, 0x61, 0x82, 0x04,
                0x4a, 0x30, 0x82, 0x04, 0x46, 0xa0, 0x03, 0x02, 0x01, 0x05, 0xa1, 0x0f, 0x1b, 0x0d, 0x43, 0x4f,
                0x4e, 0x43, 0x41, 0x56, 0x45, 0x2e, 0x4c, 0x4f, 0x43, 0x41, 0x4c, 0xa2, 0x19, 0x30, 0x17, 0xa0,
                0x03, 0x02, 0x01, 0x02, 0xa1, 0x10, 0x30, 0x0e, 0x1b, 0x04, 0x63, 0x69, 0x66, 0x73, 0x1b, 0x06,
                0x32, 0x30, 0x31, 0x32, 0x52, 0x32, 0xa3, 0x82, 0x04, 0x11, 0x30, 0x82, 0x04, 0x0d, 0xa0, 0x03,
                0x02, 0x01, 0x12, 0xa1, 0x03, 0x02, 0x01, 0x0d, 0xa2, 0x82, 0x03, 0xff, 0x04, 0x82, 0x03, 0xfb,
                0x97, 0x21, 0x22, 0x84, 0xe4, 0xb9, 0x5d, 0x5f, 0x8b, 0x6e, 0x3e, 0xa1, 0xd5, 0x8e, 0xc3, 0x09,
                0xc6, 0xc5, 0xb3, 0x7b, 0x7c, 0x38, 0xfa, 0x44, 0x2d, 0xc2, 0xf1, 0x26, 0x24, 0xd5, 0x46, 0x23,
                0xcb, 0x11, 0x5b, 0xdc, 0x4a, 0xbc, 0xdc, 0x76, 0x53, 0x1d, 0x88, 0x35, 0xf1, 0x7d, 0x41, 0x30,
                0x8b, 0xc5, 0x5a, 0x6e, 0x46, 0x01, 0x56, 0xd8, 0x13, 0xdb, 0x5c, 0xcf, 0x0f, 0xfd, 0x12, 0xa5,
                0xde, 0x96, 0x0a, 0x3d, 0xf4, 0x79, 0x44, 0x27, 0x8e, 0xd9, 0x82, 0xef, 0xe0, 0x45, 0x0f, 0x99,
                0x14, 0xfb, 0x4c, 0x78, 0xc3, 0x06, 0x8c, 0x6b, 0x81, 0x81, 0xc2, 0x21, 0xee, 0x23, 0x6f, 0x06,
                0x04, 0xf1, 0xd6, 0xc6, 0x26, 0x90, 0x50, 0x22, 0x09, 0xdc, 0x1d, 0x04, 0xf3, 0x1c, 0x45, 0x9b,
                0x72, 0xea, 0xbf, 0x77, 0x71, 0xb1, 0x94, 0xac, 0xb4, 0xd9, 0x1a, 0x8c, 0xac, 0x57, 0x78, 0xaf,
                0x51, 0x05, 0x71, 0x0b, 0x09, 0xf0, 0xbd, 0x1a, 0x43, 0x16, 0xe9, 0xdb, 0xab, 0x55, 0xfe, 0x78,
                0x16, 0x3c, 0x3e, 0x32, 0x5c, 0x3d, 0xcb, 0x62, 0xa0, 0x65, 0x49, 0x49, 0x8a, 0xb9, 0xd7, 0x77,
                0x2b, 0x2a, 0x16, 0xe5, 0x5b, 0x09, 0x36, 0x29, 0xf9, 0x83, 0x4d, 0x78, 0x5d, 0xdf, 0x97, 0xc8,
                0xfc, 0x45, 0x38, 0xbe, 0x9f, 0x1b, 0x4c, 0xfa, 0x5e, 0x64, 0xc5, 0xfc, 0x1d, 0xa8, 0x87, 0xaf,
                0xd2, 0x70, 0xd1, 0x79, 0xf3, 0xb1, 0x9f, 0x2d, 0xd0, 0x0d, 0xe0, 0xa2, 0x2a, 0xa7, 0xc3, 0xff,
                0xaf, 0xd2, 0x56, 0x82, 0x53, 0xe4, 0xf7, 0x86, 0xb4, 0x6e, 0x5e, 0x4e, 0x67, 0xe8, 0x5a, 0x35,
                0x82, 0x16, 0x94, 0x5a, 0xd2, 0x98, 0x6d, 0xa1, 0xbb, 0x8a, 0xdc, 0xec, 0xe9, 0x4f, 0x33, 0xd1,
                0x5e, 0x1b, 0xb6, 0x96, 0x59, 0xbc, 0xcb, 0x41, 0x48, 0x88, 0x6c, 0x5b, 0x05, 0xa8, 0xfb, 0x04,
                0x95, 0xf1, 0x09, 0xfd, 0x10, 0xe4, 0x3f, 0x85, 0x46, 0xe3, 0xa3, 0x1c, 0x2f, 0x6e, 0x02, 0xb4,
                0x84, 0x86, 0x4e, 0xb2, 0x1e, 0xe1, 0xf9, 0xe2, 0x4d, 0xdd, 0x3c, 0x0b, 0x69, 0xd4, 0x78, 0xb5,
                0x56, 0xac, 0x70, 0xa9, 0x25, 0x48, 0xc8, 0xdc, 0xcc, 0x6c, 0xff, 0x32, 0xd8, 0x27, 0x33, 0x68,
                0xda, 0xae, 0x7e, 0x55, 0x00, 0xd5, 0xc1, 0x92, 0x6e, 0xfb, 0x1d, 0x2b, 0x2e, 0xb2, 0xbf, 0xec,
                0x5b, 0xd5, 0xd9, 0x6d, 0x6a, 0x1c, 0x3b, 0xa9, 0x8c, 0x68, 0x4f, 0xac, 0xa0, 0x88, 0x3d, 0xfa,
                0x02, 0x6c, 0xab, 0xce, 0x17, 0x05, 0x6c, 0xde, 0x3f, 0xcf, 0x05, 0xbc, 0x80, 0xdd, 0xd2, 0x9e,
                0x33, 0x84, 0xca, 0x13, 0xae, 0xfd, 0xb2, 0x67, 0xdc, 0xab, 0x81, 0x19, 0xe9, 0xd1, 0x94, 0x0f,
                0x48, 0x90, 0x1d, 0x22, 0x90, 0xe1, 0xcc, 0x54, 0x61, 0xd7, 0xd9, 0xe4, 0x19, 0xb2, 0x9a, 0x21,
                0x47, 0x36, 0x36, 0x35, 0xd0, 0x15, 0xea, 0x51, 0xd9, 0x59, 0x1e, 0x24, 0x64, 0xc8, 0x2b, 0x17,
                0x55, 0xb4, 0x57, 0xe3, 0xcf, 0x1a, 0x18, 0x47, 0x46, 0x7b, 0x6d, 0x56, 0xaa, 0x21, 0x98, 0x35,
                0x9a, 0x6a, 0x72, 0xdc, 0x86, 0x00, 0x8c, 0x14, 0xd6, 0xb5, 0xc0, 0x41, 0x03, 0x21, 0x90, 0x27,
                0x49, 0x02, 0x78, 0x84, 0x9a, 0x57, 0x68, 0x97, 0xec, 0x73, 0xf2, 0xe1, 0x39, 0x86, 0x5b, 0x06,
                0xb9, 0xcb, 0xa5, 0x9a, 0xff, 0x5a, 0x29, 0x89, 0x68, 0x02, 0xda, 0x83, 0x6f, 0xb8, 0x1d, 0x86,
                0x36, 0xa7, 0x2d, 0xc6, 0x1e, 0x90, 0x01, 0x1a, 0x53, 0xee, 0x89, 0xbe, 0xac, 0x14, 0xe0, 0x0d,
                0x0e, 0xa3, 0x87, 0xed, 0x19, 0x23, 0x97, 0x21, 0x04, 0x60, 0xe9, 0xa8, 0x52, 0x83, 0xdb, 0xd5,
                0x84, 0x78, 0x5d, 0x29, 0x27, 0x3d, 0x70, 0x28, 0x69, 0x3b, 0x1d, 0x94, 0x98, 0x26, 0xb5, 0x69,
                0x7b, 0x61, 0xce, 0xce, 0xc2, 0x02, 0x11, 0xe6, 0x89, 0x35, 0xc7, 0xa1, 0x33, 0x9b, 0xc0, 0x23,
                0xed, 0x79, 0x96, 0x3d, 0x8f, 0xfc, 0x07, 0x9e, 0xc2, 0x84, 0x0a, 0x85, 0x5c, 0x9f, 0x9b, 0xd9,
                0xc3, 0xf6, 0x4b, 0xc9, 0xa4, 0x60, 0xfc, 0x62, 0x91, 0xfa, 0x14, 0x95, 0xaf, 0xe3, 0x64, 0xe5,
                0x11, 0x70, 0x9c, 0x18, 0x5e, 0x86, 0x3f, 0x72, 0x31, 0x1c, 0x46, 0x16, 0xc9, 0xef, 0x3b, 0xd1,
                0x2f, 0xe4, 0x7c, 0xc4, 0x6f, 0x43, 0x57, 0x26, 0x37, 0x3c, 0x28, 0x00, 0x09, 0x4f, 0x41, 0xbb,
                0x78, 0xb5, 0xb4, 0xaf, 0x5f, 0x77, 0xf5, 0x08, 0x67, 0x09, 0x1d, 0x9c, 0x75, 0x8a, 0x95, 0x50,
                0x2a, 0xd0, 0xbd, 0xa7, 0xed, 0xf1, 0xa8, 0x36, 0x27, 0x0a, 0x4d, 0x73, 0x0b, 0x95, 0xc4, 0x9f,
                0x15, 0x18, 0xaf, 0x5c, 0xd5, 0x54, 0xb9, 0x0f, 0xb9, 0xb5, 0x06, 0x87, 0xf0, 0x84, 0x83, 0x6a,
                0x35, 0xcd, 0xcf, 0x4d, 0xd6, 0xf8, 0x15, 0x2d, 0x82, 0xc9, 0x23, 0xf8, 0x62, 0xed, 0xfd, 0x8b,
                0x6b, 0x00, 0xb6, 0x7d, 0x40, 0x3e, 0xb9, 0x5c, 0x78, 0x81, 0x6d, 0xab, 0x30, 0x20, 0x38, 0x56,
                0x16, 0xd7, 0x16, 0x4c, 0x9d, 0xaa, 0x63, 0x22, 0xe3, 0x8c, 0x9a, 0xcc, 0x57, 0x0e, 0x4e, 0x46,
                0x13, 0x6f, 0xae, 0xb9, 0x73, 0x60, 0x01, 0x04, 0x7d, 0xfa, 0x51, 0x42, 0x44, 0xf6, 0x04, 0x81,
                0x42, 0x27, 0x00, 0x69, 0x39, 0xbe, 0xb0, 0x81, 0x34, 0xce, 0x2e, 0x75, 0x26, 0xe7, 0x77, 0x1b,
                0xf1, 0x94, 0xbb, 0xa3, 0x93, 0x4a, 0xca, 0x54, 0x19, 0x75, 0x24, 0xdc, 0xc2, 0xc5, 0x2e, 0xc1,
                0xb3, 0xba, 0x99, 0x83, 0xa6, 0x15, 0xda, 0x06, 0xce, 0x60, 0x89, 0x9e, 0xf9, 0x9e, 0xd6, 0x86,
                0xa9, 0xd4, 0xfe, 0xd8, 0x37, 0x99, 0xd1, 0xbe, 0xb5, 0x7f, 0x00, 0x51, 0x62, 0x8a, 0x83, 0x8a,
                0x2f, 0x4d, 0x2d, 0xcb, 0x75, 0x43, 0x1c, 0x06, 0x16, 0x15, 0x0f, 0x96, 0xd3, 0xcd, 0xc2, 0x8b,
                0x76, 0xad, 0xfc, 0xab, 0x2d, 0xbb, 0x09, 0xf2, 0x83, 0xc5, 0xb4, 0x0e, 0xdf, 0x4f, 0x61, 0x22,
                0xe5, 0x6c, 0x36, 0xa7, 0x34, 0x62, 0x2b, 0xdf, 0x8e, 0x63, 0x39, 0xfd, 0xf9, 0xd7, 0xbe, 0x7c,
                0x00, 0x34, 0xa6, 0x09, 0x16, 0x2b, 0x87, 0xe4, 0xed, 0x6c, 0xcd, 0x36, 0x3b, 0xba, 0x15, 0xdf,
                0x77, 0x88, 0xb1, 0x96, 0x56, 0x5e, 0x29, 0xfb, 0xb0, 0x21, 0xa7, 0x37, 0x92, 0x7c, 0xc2, 0x87,
                0x08, 0xe5, 0x1c, 0xf1, 0x67, 0xee, 0xc0, 0xff, 0xa9, 0x66, 0x3f, 0x13, 0x26, 0xd0, 0xbd, 0x03,
                0x40, 0x2f, 0xeb, 0x93, 0x0d, 0x3b, 0x43, 0xd7, 0x97, 0xef, 0x9a, 0x2a, 0x67, 0x80, 0x48, 0xdf,
                0x6f, 0xfb, 0x76, 0xd7, 0xc1, 0xfb, 0xa9, 0x26, 0x7e, 0x11, 0x19, 0xa5, 0x59, 0xc8, 0xac, 0x97,
                0x07, 0x9f, 0x78, 0xa2, 0x0a, 0xa6, 0x46, 0xfd, 0x8e, 0xbb, 0xaa, 0x3d, 0x3a, 0xd3, 0xf1, 0xf6,
                0x67, 0x32, 0x23, 0x65, 0x9e, 0xb7, 0x56, 0xcd, 0xca, 0xda, 0x96, 0x5d, 0x85, 0x4c, 0x1d, 0x94,
                0xa1, 0x48, 0x02, 0x2d, 0xa1, 0x2c, 0x16, 0x0d, 0x0f, 0x64, 0x5a, 0x38, 0xf9, 0x2f, 0x11, 0xfd,
                0x6f, 0x90, 0xdb, 0x36, 0xec, 0x72, 0x58, 0x5c, 0x45, 0x0d, 0x63, 0xc1, 0xa3, 0xdf, 0xf6, 0x24,
                0x4f, 0x3d, 0x85, 0x9b, 0x5c, 0xe6, 0xec, 0xab, 0x31, 0x39, 0xb7, 0x0a, 0xf8, 0xc9, 0xec, 0x85,
                0x67, 0x27, 0x16, 0x54, 0xb1, 0x8d, 0x0d, 0xf2, 0x46, 0x68, 0xd9, 0xea, 0x6b, 0x1c, 0x5b, 0x45,
                0x0f, 0x54, 0xc7, 0x98, 0xe2, 0x6b, 0xf2, 0x70, 0xa1, 0xc8, 0xdb, 0x5a, 0x66, 0x39, 0x11, 0xdc,
                0x07, 0x73, 0xe6, 0xf5, 0xab, 0xaf, 0x7d, 0x3f, 0xb2, 0x12, 0x5a, 0xa6, 0x82, 0x01, 0x33, 0x30,
                0x82, 0x01, 0x2f, 0xa0, 0x03, 0x02, 0x01, 0x12, 0xa2, 0x82, 0x01, 0x26, 0x04, 0x82, 0x01, 0x22,
                0x3c, 0x9f, 0x49, 0x2c, 0x16, 0xe2, 0x89, 0x58, 0x12, 0x7a, 0x5c, 0x26, 0xa2, 0xf5, 0x42, 0x13,
                0x9c, 0x03, 0xb8, 0xe0, 0x32, 0x00, 0x28, 0x4c, 0x0f, 0x54, 0x63, 0xcd, 0x1b, 0xc4, 0x7b, 0x26,
                0x2e, 0xc1, 0x6f, 0x09, 0xbf, 0x5f, 0xa4, 0x03, 0xb2, 0x48, 0x05, 0x9d, 0xbe, 0x49, 0xf5, 0x54,
                0x37, 0x4c, 0x47, 0xf7, 0x90, 0x78, 0xb3, 0xfb, 0x81, 0xae, 0x3f, 0x67, 0xa5, 0x95, 0x23, 0xde,
                0xe5, 0x92, 0x02, 0x25, 0x50, 0xe0, 0x2d, 0xf3, 0xaf, 0x9b, 0xf6, 0xe6, 0xa6, 0x93, 0x9f, 0x7c,
                0xdc, 0xd6, 0xdd, 0x28, 0xf4, 0x28, 0xad, 0x15, 0x8c, 0xff, 0x75, 0xc3, 0x79, 0x1a, 0x7e, 0x11,
                0x4c, 0x52, 0xd7, 0xdd, 0xfd, 0x0e, 0xb7, 0xc1, 0x65, 0x89, 0xcf, 0xca, 0x3e, 0x2f, 0x72, 0xcc,
                0xbe, 0xf6, 0x00, 0x41, 0xc4, 0x9b, 0x57, 0x1d, 0x49, 0x91, 0x44, 0x38, 0x10, 0x04, 0x99, 0x28,
                0x76, 0x0c, 0x9f, 0x4d, 0x0e, 0x08, 0xbb, 0xbf, 0xe3, 0x47, 0xea, 0x6d, 0x2e, 0x9d, 0x38, 0x4f,
                0x9d, 0xe1, 0x47, 0x47, 0x22, 0x9f, 0x13, 0x06, 0x78, 0x7c, 0x0a, 0x25, 0x77, 0x1d, 0x70, 0xb5,
                0x11, 0x12, 0xd2, 0x11, 0x65, 0xf1, 0xd2, 0x19, 0xe4, 0xf7, 0xa6, 0xcb, 0x3b, 0xff, 0x60, 0x9a,
                0x48, 0x32, 0xc2, 0x29, 0x7b, 0xc2, 0x1d, 0xc2, 0x5d, 0x90, 0x82, 0x9d, 0x74, 0xea, 0x5f, 0xfe,
                0x54, 0x40, 0xc3, 0x8b, 0xa7, 0x29, 0x66, 0xf0, 0x41, 0xb4, 0x75, 0x43, 0x65, 0xed, 0xd6, 0x69,
                0xd0, 0xb8, 0xd5, 0x97, 0xc8, 0x3d, 0x3d, 0xce, 0xb7, 0xa4, 0x9d, 0x19, 0x68, 0x46, 0x24, 0x35,
                0x87, 0x2d, 0xfd, 0x5a, 0x50, 0x77, 0xdf, 0x41, 0xab, 0xfc, 0x33, 0x13, 0x19, 0x6a, 0x95, 0x85,
                0x84, 0x86, 0xd5, 0xfd, 0x77, 0xae, 0xc4, 0xbf, 0xa1, 0x70, 0x72, 0xff, 0x0e, 0x40, 0xb5, 0x4d,
                0x6b, 0x05, 0xab, 0x8c, 0xbb, 0x08, 0x14, 0x1b, 0xd1, 0xea, 0x2a, 0x98, 0x1d, 0x7f, 0xbb, 0x88,
                0xfd, 0xc0, 0xab, 0x02, 0xf2, 0x84, 0xa0, 0x09, 0xf1, 0xdd, 0x71, 0xe9, 0x28, 0xcb, 0xaf, 0xc6,
                0x84, 0x5b
            };

            // Act.
            var kerberosTgsRepPacket = KerberosPacketParser.GetKerberosPacket(kerberosTgsRepBuffer, protocol: "TCP") as KerberosTgsRepPacket;

            // Assert.
            Assert.AreEqual(kerberosTgsRepPacket.Pvno, 5);
            Assert.AreEqual(kerberosTgsRepPacket.Crealm, "CONCAVE.LOCAL");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Realm, "CONCAVE.LOCAL");
            Assert.AreEqual(kerberosTgsRepPacket.Cname.Name, "jim");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Sname.Name, "cifs/2012R2");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Etype, 18);
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Cipher.Length, 1019);
        }

        [TestMethod]
        public void KerberosAsRepHash_Parse_ParseSuccess()
        {
            // Arrange.
            var kerberosAsRepBuffer = new byte[]
            {
                0x6b, 0x82, 0x04, 0xd5, 0x30, 0x82, 0x04, 0xd1, 0xa0, 0x03, 0x02, 0x01, 0x05, 0xa1, 0x03, 0x02, 0x01, 0x0b, 0xa3, 0x0c, 0x1b, 0x0a, 0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0x2e, 0x43, 0x4f, 0x4d,
                0xa4, 0x0f, 0x30, 0x0d, 0xa0, 0x03, 0x02, 0x01, 0x01, 0xa1, 0x06, 0x30, 0x04, 0x1b, 0x02, 0x75, 0x35, 0xa5, 0x82, 0x03, 0x73, 0x61, 0x82, 0x03, 0x6f, 0x30, 0x82, 0x03, 0x6b, 0xa0, 0x03, 0x02,
                0x01, 0x05, 0xa1, 0x0c, 0x1b, 0x0a, 0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0x2e, 0x43, 0x4f, 0x4d, 0xa2, 0x1f, 0x30, 0x1d, 0xa0, 0x03, 0x02, 0x01, 0x02, 0xa1, 0x16, 0x30, 0x14, 0x1b, 0x06, 0x6b,
                0x72, 0x62, 0x74, 0x67, 0x74, 0x1b, 0x0a, 0x44, 0x45, 0x4e, 0x59, 0x44, 0x43, 0x2e, 0x43, 0x4f, 0x4d, 0xa3, 0x82, 0x03, 0x33, 0x30, 0x82, 0x03, 0x2f, 0xa0, 0x03, 0x02, 0x01, 0x17, 0xa1, 0x03,
                0x02, 0x01, 0x02, 0xa2, 0x82, 0x03, 0x21, 0x04, 0x82, 0x03, 0x1d, 0x08, 0xca, 0x3b, 0xf3, 0x9e, 0xac, 0xf2, 0x47, 0x51, 0x55, 0xdf, 0xbb, 0x15, 0x49, 0x74, 0x3e, 0xaf, 0x43, 0xf3, 0x46, 0x9c,
                0x69, 0x81, 0x9b, 0x03, 0xa7, 0xdd, 0xb2, 0xc1, 0xa9, 0xba, 0xac, 0x1b, 0xe8, 0x3d, 0x36, 0x78, 0x7a, 0xca, 0xb8, 0xd8, 0xf5, 0xde, 0x44, 0xf0, 0x08, 0xa8, 0xb4, 0x61, 0x69, 0xa4, 0x6c, 0x7d,
                0xc4, 0xd9, 0xb0, 0x06, 0xe3, 0x48, 0x9a, 0x21, 0x71, 0x7c, 0xd9, 0x84, 0x83, 0x7b, 0x87, 0xfd, 0xd5, 0x56, 0x1d, 0x67, 0x7d, 0xab, 0xab, 0xaa, 0x37, 0x19, 0x6a, 0xe2, 0x5f, 0x41, 0x25, 0xe2,
                0x27, 0x59, 0xb9, 0x36, 0xe2, 0x07, 0xd3, 0xbe, 0x39, 0xd9, 0x3c, 0x8c, 0x5a, 0x64, 0xe5, 0xda, 0x3c, 0xa2, 0x06, 0x58, 0xa1, 0x28, 0x3f, 0x07, 0xd4, 0xbc, 0x6f, 0xda, 0x6e, 0x61, 0xce, 0x26,
                0xb2, 0x79, 0x74, 0xca, 0x83, 0x8d, 0x79, 0xf5, 0xd8, 0x6a, 0xbc, 0xaa, 0xd9, 0x43, 0x80, 0xef, 0x5b, 0x30, 0x48, 0xa4, 0x8c, 0x6a, 0x25, 0xfd, 0x78, 0x0e, 0x10, 0x4d, 0xc0, 0xa8, 0x6b, 0x83,
                0x40, 0xe6, 0x0d, 0xf4, 0x00, 0xfa, 0x29, 0xca, 0x1d, 0x5d, 0xfe, 0x65, 0x8f, 0x20, 0x69, 0xb7, 0x08, 0xf1, 0xc4, 0x44, 0xa3, 0xfd, 0x2e, 0xca, 0xcf, 0xa8, 0x7d, 0x42, 0x52, 0x57, 0xfd, 0xb3,
                0xc5, 0xeb, 0x9b, 0xba, 0xf4, 0x70, 0x85, 0x0b, 0xc0, 0xf4, 0x3b, 0xac, 0x7a, 0x49, 0x83, 0xe9, 0x25, 0x0c, 0x20, 0x17, 0x9d, 0x15, 0x1f, 0x39, 0xf5, 0xf8, 0xdf, 0x66, 0x9c, 0x5c, 0xe7, 0x69,
                0x86, 0xb8, 0x02, 0xe4, 0xf4, 0x1f, 0x5e, 0x9b, 0x63, 0xc6, 0x56, 0x93, 0x79, 0x9e, 0x98, 0x75, 0x1d, 0xa4, 0xa5, 0xdb, 0x2c, 0x3c, 0xc3, 0xa7, 0xa7, 0xa6, 0x32, 0x9d, 0x9e, 0x5c, 0xff, 0x5e,
                0x9b, 0x12, 0x45, 0x15, 0x05, 0x12, 0xf0, 0xcd, 0x1f, 0xe7, 0x9f, 0x3a, 0x88, 0x1e, 0x3d, 0x10, 0x14, 0x52, 0xda, 0xcd, 0x84, 0x78, 0xf7, 0xcf, 0x59, 0xc4, 0xbe, 0x98, 0xde, 0x83, 0x61, 0xa0,
                0x60, 0xdf, 0x9d, 0xc0, 0xc1, 0x87, 0xb3, 0xd4, 0xb8, 0x01, 0xec, 0xfb, 0xb0, 0xd8, 0xd3, 0x4f, 0x3c, 0x31, 0x65, 0x56, 0x36, 0x14, 0xcb, 0x25, 0xc7, 0xd0, 0x7c, 0xb1, 0x3f, 0x40, 0xc4, 0x5e,
                0x9b, 0x90, 0x10, 0x2f, 0xeb, 0x76, 0x42, 0x3a, 0xc2, 0x49, 0xd0, 0x87, 0xa1, 0xbd, 0x96, 0x61, 0xa4, 0xd9, 0x5f, 0x35, 0x5a, 0x07, 0x59, 0x95, 0x29, 0x3e, 0x0a, 0x48, 0x8b, 0xe7, 0xaa, 0x01,
                0xb4, 0xc1, 0x82, 0x8a, 0x54, 0x43, 0x56, 0x10, 0x28, 0xa8, 0x2d, 0x2f, 0x27, 0xf9, 0x06, 0x7f, 0xa5, 0xf1, 0x4a, 0x8b, 0x6a, 0x2a, 0x48, 0x32, 0xed, 0x5a, 0xe0, 0x22, 0x24, 0x53, 0xe5, 0x5f,
                0x71, 0x8e, 0xdd, 0x54, 0x0f, 0xc3, 0xb0, 0x93, 0xd3, 0xca, 0x40, 0x8d, 0x6c, 0x88, 0x4b, 0xe5, 0xc8, 0x39, 0x31, 0x62, 0xb1, 0x2b, 0xea, 0xfc, 0xf2, 0x19, 0xd8, 0xc4, 0x11, 0xd9, 0xf5, 0x27,
                0x6f, 0xd1, 0xa9, 0x54, 0xfa, 0x05, 0x59, 0xcb, 0x65, 0xe2, 0x1c, 0xea, 0xe6, 0x4d, 0x62, 0x87, 0x50, 0x64, 0x36, 0xc2, 0x95, 0xe3, 0xe7, 0xfe, 0xd8, 0x81, 0xcd, 0x62, 0x11, 0x7c, 0x00, 0x01,
                0x17, 0x33, 0xda, 0x0e, 0xa2, 0x05, 0x72, 0xe5, 0xaf, 0x95, 0xb9, 0xd7, 0xb6, 0x0a, 0x3d, 0xa2, 0x4f, 0x86, 0x65, 0xa8, 0x0c, 0xea, 0xd5, 0x50, 0x49, 0x80, 0x36, 0xa0, 0x8f, 0x0e, 0xba, 0xad,
                0xcd, 0x22, 0x58, 0x4a, 0x3c, 0x4e, 0x7e, 0x00, 0x18, 0x06, 0xf1, 0x00, 0xd5, 0x0d, 0x6f, 0x71, 0x29, 0x1e, 0xa3, 0x6b, 0x39, 0x78, 0x8f, 0x0c, 0xde, 0xda, 0xcc, 0xa5, 0xfc, 0xc5, 0x1e, 0xc3,
                0x8c, 0x56, 0xd2, 0xd1, 0x02, 0xc4, 0xb1, 0x28, 0x2d, 0xaf, 0x4d, 0x6a, 0x4a, 0xa0, 0x74, 0x04, 0xc3, 0x98, 0xd5, 0x05, 0xa4, 0x93, 0x76, 0xf1, 0xe9, 0xc3, 0x51, 0x09, 0xb2, 0x44, 0x7c, 0x4b,
                0x65, 0x7a, 0x00, 0xf7, 0xfa, 0x2c, 0xa7, 0x0b, 0x5a, 0xf2, 0xa9, 0x55, 0xd8, 0xfc, 0x02, 0xa1, 0x54, 0xfa, 0x46, 0x05, 0xa0, 0x4e, 0xe4, 0x09, 0xdc, 0xc8, 0x82, 0xde, 0xdd, 0x31, 0x7b, 0x82,
                0xad, 0xb4, 0x5e, 0x81, 0x43, 0x3c, 0xce, 0xd8, 0xd2, 0x10, 0xb6, 0x78, 0xac, 0xe6, 0x0e, 0x98, 0xf8, 0xd0, 0x3e, 0xe2, 0x7d, 0xff, 0xe8, 0x2d, 0x3a, 0x0a, 0x8f, 0x3c, 0x85, 0x23, 0xf5, 0x18,
                0x08, 0xc7, 0xac, 0xbe, 0xe5, 0xb8, 0xf3, 0x5d, 0xf1, 0xe9, 0x29, 0x41, 0xb7, 0x78, 0x54, 0xdf, 0xf8, 0x68, 0x1c, 0x29, 0x37, 0x23, 0x43, 0xe5, 0x42, 0x49, 0xa7, 0x75, 0x1f, 0xa2, 0x29, 0x17,
                0x62, 0x86, 0x63, 0x3e, 0xcb, 0x47, 0x90, 0xf1, 0xab, 0xba, 0x8a, 0x89, 0x6c, 0xc0, 0x33, 0xbc, 0x46, 0xf0, 0xd1, 0x81, 0xe4, 0xe4, 0xe6, 0xb0, 0x01, 0xb1, 0x01, 0xa2, 0xfc, 0x5d, 0x0d, 0x39,
                0xad, 0xd2, 0x1a, 0xb6, 0x81, 0x35, 0xa9, 0x3f, 0x28, 0x56, 0x51, 0x16, 0x29, 0x91, 0x02, 0xbd, 0xea, 0x07, 0xe3, 0x58, 0xd6, 0x18, 0x8d, 0xdb, 0xbf, 0xca, 0xa0, 0x46, 0x78, 0xde, 0xfd, 0x72,
                0xdf, 0xb6, 0x31, 0x23, 0x6b, 0x59, 0xd6, 0x31, 0x73, 0xc9, 0x3d, 0xec, 0xb1, 0x5d, 0x13, 0xc3, 0x3c, 0xb2, 0xda, 0xe5, 0x05, 0x8b, 0x25, 0xc8, 0xd6, 0x80, 0xa3, 0x5d, 0x2d, 0x51, 0xa6, 0x47,
                0xc5, 0xed, 0x92, 0xd4, 0xd5, 0x34, 0x3e, 0x37, 0xbc, 0x5a, 0x2c, 0x63, 0xb2, 0x92, 0x0c, 0x06, 0x9a, 0xf1, 0xb0, 0x37, 0x5c, 0xf9, 0x00, 0x36, 0x8d, 0x5d, 0xb2, 0x16, 0x0c, 0x46, 0x44, 0x00,
                0x22, 0xc6, 0x99, 0x7b, 0x8d, 0x79, 0x90, 0x4c, 0x70, 0xbc, 0x48, 0x93, 0x3c, 0x76, 0x9e, 0x6d, 0x18, 0x03, 0xdd, 0x77, 0x77, 0x6c, 0xd6, 0xd7, 0x34, 0xf9, 0xe8, 0x74, 0x42, 0x75, 0x0b, 0x1a,
                0x8e, 0x30, 0x37, 0x32, 0x2f, 0x57, 0x24, 0x5e, 0xa6, 0x82, 0x01, 0x2d, 0x30, 0x82, 0x01, 0x29, 0xa0, 0x03, 0x02, 0x01, 0x17, 0xa1, 0x03, 0x02, 0x01, 0x07, 0xa2, 0x82, 0x01, 0x1b, 0x04, 0x82,
                0x01, 0x17, 0x0f, 0x4f, 0xb3, 0x54, 0x5e, 0x75, 0x5b, 0x1d, 0x02, 0xc4, 0x17, 0x43, 0xa1, 0x0f, 0x5a, 0xf9, 0x43, 0x2a, 0xcc, 0xa6, 0x0f, 0x01, 0xdb, 0x13, 0x2d, 0xfb, 0x98, 0x01, 0xb3, 0xe3,
                0xd7, 0xb2, 0x61, 0x58, 0x37, 0xdb, 0x36, 0x1c, 0x5a, 0xd4, 0x6a, 0x7b, 0x24, 0x4b, 0x08, 0xf2, 0x46, 0xf7, 0x55, 0x37, 0xba, 0xf4, 0x2e, 0xf5, 0x47, 0x2e, 0xca, 0xbd, 0xc0, 0xcb, 0x85, 0x82,
                0x49, 0xda, 0x43, 0x23, 0x5a, 0x5b, 0xed, 0x30, 0xbe, 0x29, 0xe0, 0x94, 0x41, 0x56, 0x84, 0x67, 0x5a, 0xe4, 0x9d, 0x03, 0xa4, 0x5f, 0xf0, 0xc7, 0x9b, 0x7b, 0xf3, 0x58, 0xca, 0xb5, 0x3d, 0xc4,
                0x21, 0x4e, 0x27, 0xee, 0x83, 0x60, 0x7f, 0xa9, 0x20, 0x37, 0xe7, 0x5c, 0xe4, 0x8a, 0x3d, 0xfe, 0x02, 0x69, 0x43, 0x3f, 0x5d, 0xe0, 0xfa, 0x08, 0xd1, 0x27, 0xd8, 0x68, 0x39, 0xc1, 0xa9, 0xed,
                0xa5, 0x14, 0xa8, 0x1d, 0xc7, 0xc8, 0x18, 0x8e, 0x6d, 0xdc, 0x3c, 0xab, 0x01, 0xdd, 0xe1, 0xeb, 0xb9, 0x55, 0x74, 0x2e, 0xf1, 0x81, 0x9e, 0xdf, 0x48, 0xca, 0x50, 0x20, 0x86, 0x27, 0x61, 0xe9,
                0xac, 0x2d, 0xed, 0x29, 0x01, 0xe5, 0x90, 0x27, 0xae, 0x28, 0xa1, 0x8b, 0x05, 0xd6, 0x68, 0xd2, 0x3f, 0xae, 0xaf, 0x9d, 0x42, 0x38, 0xb6, 0x83, 0x07, 0x1e, 0x56, 0x62, 0x59, 0x34, 0x29, 0x35,
                0x7f, 0x2a, 0xe4, 0xfd, 0x32, 0x6c, 0xfb, 0x2b, 0xb3, 0xe6, 0xeb, 0x8d, 0x77, 0xd1, 0xf5, 0x9d, 0xbf, 0xbd, 0x76, 0xc7, 0x0c, 0x17, 0xc7, 0xe4, 0xe2, 0xf7, 0x83, 0xfe, 0xe0, 0xe5, 0x77, 0x05,
                0x7a, 0xc5, 0x7d, 0xf2, 0xb0, 0x99, 0xe0, 0x18, 0xda, 0xea, 0x86, 0xdd, 0x2b, 0xae, 0x3f, 0x5b, 0xa7, 0xd7, 0x0b, 0x09, 0x1b, 0xd6, 0x6d, 0x8b, 0xbf, 0x39, 0xcb, 0x96, 0x03, 0x9f, 0xd6, 0x31,
                0x20, 0xf0, 0x18, 0x3f, 0x89, 0x43, 0xbd, 0xaa, 0x9f, 0x0b, 0x83, 0x7e, 0xe8, 0x40, 0xa1, 0x61, 0x64, 0xc2, 0x42, 0x26, 0xc2, 0x06, 0xd4, 0x9a, 0x46
            };

            // Act.
            var kerberosTgsRepPacket = KerberosPacketParser.GetKerberosPacket(kerberosAsRepBuffer, protocol:"UDP") as KerberosAsRepPacket;

            // Assert.
            Assert.AreEqual(kerberosTgsRepPacket.Pvno, 5);
            Assert.AreEqual(kerberosTgsRepPacket.Crealm, "DENYDC.COM");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Realm, "DENYDC.COM");
            Assert.AreEqual(kerberosTgsRepPacket.Cname.Name, "u5");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.Sname.Name, "krbtgt/DENYDC.COM");
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Etype, 23);
            Assert.AreEqual(kerberosTgsRepPacket.Ticket.EncrytedPart.Cipher.Length, 797);
        }

        private PcapAnalyzer.TcpPacket mockPacket(string sourceIp, string destinationIp, int sourcePort, int destinationPort, string data)
        {
            return new PcapAnalyzer.TcpPacket()
            {
                SourceIp = sourceIp,
                DestinationIp = destinationIp,
                SourcePort = sourcePort,
                DestinationPort = destinationPort,
                Data = Encoding.ASCII.GetBytes(data)
            };
        }
    }
}
